import win32api, win32gui, win32ui, win32con
import cv2
import numpy as np
from PIL import Image
import os
import rapidocr_onnxruntime 

#返回标题不为空的所有窗口
def get_all_windows():
    hwnd_title = {}
    def callback(hwnd, hwnd_title):
        if win32gui.IsWindow(hwnd) and win32gui.IsWindowEnabled(hwnd):
            title = win32gui.GetWindowText(hwnd)
            if title != "":
                hwnd_title[title] = hwnd
    win32gui.EnumWindows(callback, hwnd_title)
    return hwnd_title

#返回标题内包含title的窗口句柄
def get_window_by_title(title=""):
    for htitle, hwnd in get_all_windows().items():
        if htitle.find(title) > -1:
            return hwnd
    return None

#激活标题包含title的窗口，设置为前端显示
def active_window_by_title(title=""):
    if title!="":
        hwnd = get_window_by_title(title)
        if hwnd != None:
            win32gui.ShowWindow(hwnd,4)
            win32gui.SetForegroundWindow(hwnd)

# 获取标题是title的窗口的坐标
def get_rect_for_title(title=""):
    # 找到窗口的句柄
    if title!="":
        hwnd = win32gui.FindWindow(None, title)
    if not hwnd:
        raise Exception("Window not found!")
    # 获取窗口的坐标
    rect = win32gui.GetWindowRect(hwnd)
    return rect

# 获取标题包含title的窗口坐标
def get_rect_by_title(title=""):
    if title!="":
        hwnd = get_window_by_title(title)
        if hwnd != None:
            return win32gui.GetWindowRect(hwnd)

# 获取句柄是hwnd的窗口坐标
def get_rect_by_hwnd(hwnd):
    if hwnd!=None:
        return win32gui.GetWindowRect(hwnd)

# 通过窗口句柄截取当前句柄图片 返回cv2格式的Mat数据
def window_capture(hwnd, picture_name=None):
    x1, y1, x2, y2 = win32gui.GetWindowRect(hwnd)  # 获取当前窗口大小
    hwndDC = win32gui.GetWindowDC(hwnd)  # 通过应用窗口句柄获得窗口DC
    mfcDC = win32ui.CreateDCFromHandle(hwndDC)  # 通过hwndDC获得mfcDC(注意主窗口用的是win32gui库，操作位图截图是用win32ui库)
    neicunDC = mfcDC.CreateCompatibleDC()  # 创建兼容DC，实际在内存开辟空间（ 将位图BitBlt至屏幕缓冲区（内存），而不是将屏幕缓冲区替换成自己的位图。同时解决绘图闪烁等问题）
    savebitmap = win32ui.CreateBitmap()  # 创建位图
    width = x2 - x1
    height = y2 - y1
    savebitmap.CreateCompatibleBitmap(mfcDC, width, height)  # 设置位图的大小以及内容
    neicunDC.SelectObject(savebitmap)  # 将位图放置在兼容DC，即 将位图数据放置在刚开辟的内存里
    neicunDC.BitBlt((0, 0), (width, height), mfcDC, (0, 0), win32con.SRCCOPY)  # 截取位图部分，并将截图保存在剪贴板
    if picture_name is not None:
        savebitmap.SaveBitmapFile(neicunDC, picture_name)  # 将截图数据从剪贴板中取出，并保存为bmp图片
        img_buf = savebitmap.GetBitmapBits(True)
        img = np.frombuffer(img_buf, dtype="uint8")
        img.shape = (height, width, 4)
        mat_img = cv2.cvtColor(img, cv2.COLOR_RGB2RGBA)  # 转换RGB顺序
        cv2.imwrite(picture_name.replace(".bmp",".jpg"), img)
    # 释放内存
    win32gui.DeleteObject(savebitmap.GetHandle())
    neicunDC.DeleteDC()
    mfcDC.DeleteDC()
    win32gui.ReleaseDC(hwnd, hwndDC)

if __name__ == '__main__':
    # 获取应用句柄
    hwnd = get_window_by_title("EVA - 车辆信息")
    print(win32gui.GetWindowRect(hwnd))
    window_capture(hwnd,"demo.bmp")
    config_path = os.path.join(os.path.dirname(rapidocr_onnxruntime.__file__),"config.yaml")
    model = rapidocr_onnxruntime.RapidOCR(config_path=config_path)
    image = Image.open("demo.jpg")
    data, _ = model( image , use_det=True, use_cls=False, use_rec=True)
    # data: List[List[float], str, float] ([[左上, 右上, 右下, 左下], 文本内容, 置信度])
    print(data)